<?php
/**
 * Create Contact
 *
 * @package     AutomatorWP\Integrations\QR_Code\Actions\Generate_QR_Code
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */

 // Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_QR_Code_Generate extends AutomatorWP_Integration_Action {

    public $integration = 'qr_code';
    public $action = 'qr_code_generate';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Generate a QR code', 'automatorwp-pro' ),
            'select_option'     => __( 'Generate a <strong>QR code</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Contact. */
            'edit_label'        => sprintf( __( 'Generate a %1$s', 'automatorwp-pro' ), '{code}' ),
            /* translators: %1$s: Contact. */
            'log_label'         => sprintf( __( 'Generate a %1$s', 'automatorwp-pro' ), '{code}' ),
            'options'           => array(
                'code' => array(
                    'default' => __( 'QR code', 'automatorwp-pro' ),
                    'fields' => array(
                        'qr_content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Content displayed by QR code.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'qr_size' => array(
                            'name' => __( 'Size:', 'automatorwp-pro' ),
                            'desc' => __( 'Size of the QR code. Default 200 px.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '200',
                            'classes' => 'automatorwp-col-3',
                        ),
                        'qr_error_correction_level' => array(
                            'name' => __( 'Error Correction Level:', 'automatorwp-pro' ),
                            'desc' => __( 'The error correction level. By default: Low.', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'low'       => __( 'Low', 'automatorwp-pro' ),
                                'medium'    => __( 'Medium', 'automatorwp-pro' ),
                                'quartile'  => __( 'Quartile', 'automatorwp-pro' ),
                                'high'      => __( 'High', 'automatorwp-pro' ),
                            ),
                            'default' => 'low',
                            'classes' => 'automatorwp-col-3',
                        ),
                        'qr_format' => array(
                            'name' => __( 'Format:', 'automatorwp-pro' ),
                            'desc' => __( 'The QR code format. By default: PNG.', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'png'   => __( 'PNG', 'automatorwp-pro' ),
                                'svg'   => __( 'SVG', 'automatorwp-pro' ),
                                'eps'   => __( 'EPS', 'automatorwp-pro' ),
                            ),
                            'default' => 'png',
                            'classes' => 'automatorwp-col-3',
                        ),
                        'qr_set_colors' => array(
                            'name' => __( 'Color Mode', 'automatorwp-pro' ),
                            'type' => 'radio_inline',
                            'options' => array(
                                'single'    => __( 'Single color', 'automatorwp-pro' ),
                                'gradient'  => __( 'Gradient', 'automatorwp-pro' ),
                            ),
                            'default' => 'single'
                        ),
                        'qr_foreground_color_1' => array(
                            'name' => __( 'Foreground Color:', 'automatorwp-pro' ),
                            //'desc' => __( 'The foreground color. RGB format. By default 0, 0, 0.', 'automatorwp-pro' ),
                            'type' => 'colorpicker',
                            'options' => array( 'alpha' => true ),
                            'default' => '#000000',
                            'classes' => 'automatorwp-col-3',
                        ),
                        'qr_foreground_color_2' => array(
                            'name' => __( 'Foreground Color 2:', 'automatorwp-pro' ),
                            //'desc' => __( 'The foreground color. RGB format. By default 0, 0, 0.', 'automatorwp-pro' ),
                            'type' => 'colorpicker',
                            'options' => array( 'alpha' => true ),
                            'default' => '#000000',
                            'classes' => 'automatorwp-col-3',
                        ),
                        'qr_eye_colors' => array(
                            'name' => __( 'Custom Eye color', 'automatorwp-pro' ),
                            'type' => 'checkbox',
                            'classes' => 'cmb2-switch'
                        ),
                        'qr_foreground_eye_color_1' => array(
                            'name' => __( 'Eye Color 1:', 'automatorwp-pro' ),
                            //'desc' => __( 'The foreground color. RGB format. By default 0, 0, 0.', 'automatorwp-pro' ),
                            'type' => 'colorpicker',
                            'options' => array( 'alpha' => true ),
                            'default' => '#000000',
                            'classes' => 'automatorwp-col-3',
                        ),
                        'qr_foreground_eye_color_2' => array(
                            'name' => __( 'Eye Color 2:', 'automatorwp-pro' ),
                            //'desc' => __( 'The foreground color. RGB format. By default 0, 0, 0.', 'automatorwp-pro' ),
                            'type' => 'colorpicker',
                            'options' => array( 'alpha' => true ),
                            'default' => '#000000',
                            'classes' => 'automatorwp-col-3',
                        ),
                        'qr_background_color' => array(
                            'name' => __( 'Background Color:', 'automatorwp-pro' ),
                            'desc' => __( 'The background color. RGB format. By default 255, 255, 255.', 'automatorwp-pro' ),
                            'type' => 'colorpicker',
                            'options' => array( 'alpha' => true ),
                            'default' => '#ffffff',
                        ),
                     ) )
            ),
            'tags'  => automatorwp_qr_code_get_actions_qr_code_tags()
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        require_once AUTOMATORWP_QR_CODE_DIR . 'vendor/autoload.php';
        
        $this->result = '';

        // Shorthand
        $qr_content = $action_options['qr_content'];
        $qr_size = $action_options['qr_size'];
        
        if( ! extension_loaded('gd') ) {
            $this->result = __( 'Extension GD not found', 'automatorwp-pro' );
            return false;
        }
        

        // Ensure that the QRs directory exists
        if( ! is_dir( AUTOMATORWP_QR_CODE_UPLOAD_DIR ) ) {
            mkdir( AUTOMATORWP_QR_CODE_UPLOAD_DIR, 755, true );
        }

        // Add an index.php file to prevent directory browsing
        if ( ! file_exists( AUTOMATORWP_QR_CODE_UPLOAD_DIR . '/index.php' ) ) {
            file_put_contents( AUTOMATORWP_QR_CODE_UPLOAD_DIR . '/index.php', '<?php // Silence is golden and GamiPress is the best plugin ever!' );
        }

        $qr_code_id = uniqid( 'automatorwp_qr_code_', true );

        if ( ! extension_loaded('imagick') ){

            $qr_code_path = AUTOMATORWP_QR_CODE_UPLOAD_DIR . $qr_code_id . '.png';

            // To generate standard QR code
            automatorwp_qr_code_generate_standard_qr( $qr_content, $qr_size, $qr_code_path );
            $this->result = __( 'Extension ImageMagick not found. Standard QR code was generated.', 'automatorwp-pro' );
            
        } else {

            // Shorthand
            $qr_format = $action_options['qr_format'];
            $qr_set_colors = $action_options['qr_set_colors'];
            $qr_set_eye = $action_options['qr_eye_colors'];
            
            // Foreground color selection 1. For single color
            if ( ! empty( $action_options['qr_foreground_color_1'] ) ) {
                $qr_foreground_color_1 = automatorwp_qr_code_hex_to_rgb( $action_options['qr_foreground_color_1'] );
            } else {
                $qr_foreground_color_1 = array( 'red' => 0, 'green' => 0, 'blue' => 0 );
            }

            // Foreground color selection 2. For gradient color
            if ( ! empty( $action_options['qr_foreground_color_2'] ) ) {
                $qr_foreground_color_2 = automatorwp_qr_code_hex_to_rgb( $action_options['qr_foreground_color_2'] );
            } else {
                $qr_foreground_color_2 = array( 'red' => 0, 'green' => 0, 'blue' => 0 );
            }

            // Foreground eye color selection 1. For gradient color
            if ( ! empty( $action_options['qr_foreground_eye_color_1'] ) ) {
                $qr_foreground_eye_color_1 = automatorwp_qr_code_hex_to_rgb( $action_options['qr_foreground_eye_color_1'] );
            } else {
                $qr_foreground_eye_color_1 = array( 'red' => 0, 'green' => 0, 'blue' => 0 );
            }

            // Foreground eye color selection 2. For gradient color
            if ( ! empty( $action_options['qr_foreground_eye_color_2'] ) ) {
                $qr_foreground_eye_color_2 = automatorwp_qr_code_hex_to_rgb( $action_options['qr_foreground_eye_color_2'] );
            } else {
                $qr_foreground_eye_color_2 = array( 'red' => 0, 'green' => 0, 'blue' => 0 );
            }
                
            // Background color selection
            if ( ! empty( $action_options['qr_background_color'] ) ) {
                $qr_background_color = automatorwp_qr_code_hex_to_rgb( $action_options['qr_background_color'] );
            } else {
                $qr_background_color = array( 'red' => 255, 'green' => 255, 'blue' => 255 );
            }

            // Get the correction error level
            $qr_error_level = automatorwp_qr_code_get_correction_level( $action_options['qr_error_correction_level'] );
            
            $qr_code_path = AUTOMATORWP_QR_CODE_UPLOAD_DIR . $qr_code_id . '.' . $qr_format;

            // If the QR code does not exists, generate a new one
            if( ! file_exists( $qr_code_path ) ) {
                
                switch ( $qr_set_colors ){
                    case 'single':
                        // To generate QR code single color
                        if( (bool) $qr_set_eye ) {
                            automatorwp_qr_code_generate_gradient_color( $qr_content, $qr_size, $qr_foreground_color_1, $qr_foreground_color_1, $qr_foreground_eye_color_1, $qr_foreground_eye_color_2, $qr_background_color, $qr_error_level, $qr_format, $qr_code_path, $qr_set_eye );
                        } else {
                            automatorwp_qr_code_generate_gradient_color( $qr_content, $qr_size, $qr_foreground_color_1, $qr_foreground_color_1, $qr_foreground_color_1, $qr_foreground_color_1, $qr_background_color, $qr_error_level, $qr_format, $qr_code_path, $qr_set_eye );
                        }
                        break;
                    case 'gradient':
                        // To generate QR code gradient color
                        automatorwp_qr_code_generate_gradient_color( $qr_content, $qr_size, $qr_foreground_color_1, $qr_foreground_color_2, $qr_foreground_eye_color_1, $qr_foreground_eye_color_2, $qr_background_color, $qr_error_level, $qr_format, $qr_code_path, $qr_set_eye );
                        break;
                    
                }
            
            }

            $this->result = __( 'QR code was generated.', 'automatorwp-pro' );
        }

        // Tags
        if ( !isset ( $qr_format) )
            $qr_format = 'png';

        $this->qr_code_url = AUTOMATORWP_QR_CODE_UPLOAD_URL . $qr_code_id . '.' . $qr_format;
        $this->qr_code_image = sprintf( '<img src="%s" />',$this->qr_code_url );

        $this->qr_result = sprintf( __( '%s', 'automatorwp-pro' ), $this->qr_code_url );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['qr_code_url'] = ( isset( $this->qr_code_url ) ? $this->qr_code_url : '' );
        $log_meta['qr_code_image'] = ( isset( $this->qr_code_image ) ? $this->qr_code_image : '' );
        $log_meta['qr_result'] = ( isset( $this->qr_result ) ? $this->qr_result : '' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['qr_result'] = array(
            'name' => __( 'QR:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['qr_code_image'] = array(
            'name' => __( 'QR Code:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_QR_Code_Generate();